/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * graph.cc
 * Copyright (C) Mike Crash 2008 <mike@mikecrash.com>
 *
 * main.cc is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * main.cc is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <string.h>
#include <math.h>
#include "graph.h"
#include <cairomm/context.h>
#include <pangomm/layout.h>

CGraph::CGraph()
{
	offset_x = 30;
	offset_y = 15;
	min_y = 0;
	max_y = 56;
	min_x = 0;
	max_x = 750;
	size = 0;
	count = 0;
	limit = 0;
}

CGraph::~CGraph()
{
}

void CGraph::draw(Cairo::RefPtr<Cairo::Context> cr, float *buf, int cnt, int total)
{
    Gtk::Allocation allocation = get_allocation();

	if (buf == NULL) return;
	
    int xc, yc;
	const int width = allocation.get_width();
    const int height = allocation.get_height();
    cr->set_line_width(1.5);
	cr->set_source_rgb(0.8, 0.0, 0.0);
	
	yc = (height-offset_y) - (buf[0] - min_y) * (height-offset_y) / (max_y - min_y);
	cr->move_to(offset_x, yc);
	
	for (int i=1;i<cnt;i++)
	{
		xc = offset_x + i * (width-offset_x) / total;
		yc = (height-offset_y) - (buf[i] - min_y) * (height-offset_y) / (max_y - min_y);
		cr->line_to(xc, yc);
	}
	cr->stroke();
}

void CGraph::draw_axes(Cairo::RefPtr<Cairo::Context> cr)
{
	int i,num;
	float f,ff,rnd;
    Gtk::Allocation allocation = get_allocation();
	Glib::RefPtr<Pango::Layout> pangoLayout = Pango::Layout::create (cr);
	Pango::FontDescription font_descr( "sans 8" );
	const int width = allocation.get_width();
    const int height = allocation.get_height();

	offset_y = 12;
	offset_x = 0;
	
	//draw y axes
	num = (height-offset_y-20) / 70;
	if (num<1) num = 1;
	f = (abs(max_y-min_y) / (float)num);
	if (f<0.0001) f= 0.0001;
	ff = floor((log(f))/(log(10.0)));
	ff = pow(10, ff);
	f = floor(f/ff);
	if (f>5) f = 5;
	else if (f>2) f = 2;
	else f = 1;
	rnd = f * ff;
	cr->set_line_width(0.5);
	cr->set_source_rgb(0.0, 0.0, 0.8);
	
	//draw y text and determine text width
	for (f = min_y+rnd;f<max_y;f+=rnd)
	{
		int yc;
		int w,h;
		char s[50];
		yc = height-offset_y - f * (height-offset_y)/(max_y-min_y);
		//printf("%d - %f,%d\n",i,f,yc);
		
		if (unit_y && (f+rnd>=max_y))
			sprintf(s,"%g%s",f, unit_y);
		else
			sprintf(s,"%g",f);
		
	    pangoLayout->set_font_description( font_descr);		
		pangoLayout->set_text(s);
		pangoLayout->get_pixel_size(w,h);
		if (w+3>offset_x) offset_x = w+3;
		cr->move_to(0,yc-h/2);
		pangoLayout->update_from_cairo_context(cr);  //gets cairo cursor position
		pangoLayout->add_to_cairo_context(cr);       //adds text to cairos stack of stuff to be drawn
		cr->fill();                                  //tells Cairo to render it's stack	
	}

	//and lines
	for (f = min_y+rnd;f<max_y;f+=rnd)
	{
		int yc;
		int w,h;
		char s[50];
		yc = height-offset_y - f * (height-offset_y)/(max_y-min_y);
		//printf("%d - %f,%d\n",i,f,yc);
		cr->move_to(offset_x, yc);
		cr->line_to(width, yc);
		cr->stroke();
	}
	
	//draw x axes
	num = (width-offset_x-50) / 100;
	if (num<1) num = 1;
	f = (abs(max_x-min_x) / (float)num);
	if (f<0.0001) f= 0.0001;
	ff = floor((log(f))/(log(10.0)));
	ff = pow(10, ff);
	f = floor(f/ff);
	if (f>5) f = 5;
	else if (f>2) f = 2;
	else f = 1;
	rnd = f * ff;
	cr->set_line_width(0.5);
	cr->set_source_rgb(0.0, 0.0, 0.8);
	
	for (f = min_x+rnd;f<max_x;f+=rnd)
	{
		int xc;
		int w,h;
		char s[50];
		xc = offset_x + f * (width-offset_x)/(max_x-min_x);

		if (unit_x && (f+rnd>=max_x))
			sprintf(s,"%g%s",f, unit_x);
		else
			sprintf(s,"%g",f);
		
	    pangoLayout->set_font_description( font_descr);		
		pangoLayout->set_text(s);
		pangoLayout->get_pixel_size(w,h);
		cr->move_to(xc-w/2,height-offset_y);
		pangoLayout->update_from_cairo_context(cr);  //gets cairo cursor position
		pangoLayout->add_to_cairo_context(cr);       //adds text to cairos stack of stuff to be drawn
		cr->fill();                                  //tells Cairo to render it's stack	
		
		cr->move_to(xc,height-offset_y);
		cr->line_to(xc, 0);
		cr->stroke();
	}

	//draw y axes
	for (f = min_y+rnd;f<max_y;f+=rnd)
	{
		int yc;
		int w,h;
		char s[50];
		yc = height-offset_y - f * (height-offset_y)/(max_y-min_y);
		//printf("%d - %f,%d\n",i,f,yc);
		cr->move_to(offset_x, yc);
		cr->line_to(width, yc);
		cr->stroke();
	}
	
	//draw main axes
	cr->set_line_width(1.0);
	cr->set_source_rgb(0.0, 0.0, 0.8);
	cr->move_to(offset_x, height-offset_y);
    cr->line_to(offset_x, 0);
    cr->line_to(width, 0);
    cr->line_to(width, height-offset_y);
    cr->line_to(offset_x, height-offset_y);
	cr->stroke();

	if ((limit>0)&&(size>0))
	{
		int xc;
		cr->set_source_rgb(0.5, 0.0, 0.0);
		cr->set_line_width(0.5);
		xc = offset_x + limit * (width-offset_x) / size;
		cr->move_to(xc, height-offset_y);
		cr->line_to(xc, 0);
		cr->stroke();
	}
}

void CGraph::draw_graph(float *buf, int cnt, int total)
{
  //copy the buffer
  buffer = buf;
  count = cnt;
  size = total;

  //initiate redraw of graph
  Glib::RefPtr<Gdk::Window> win = get_window();
  if (win)
  {
	Gdk::Rectangle r(0, 0, get_allocation().get_width(), get_allocation().get_height());
	win->invalidate_rect(r, false);
  }
}

void CGraph::set_limits(float x1, float x2, float y1, float y2)
{
	min_x = x1;
	max_x = x2;
	min_y = y1;
	max_y = y2;
}

void CGraph::set_units(const char *x, const char *y)
{
	if (unit_x)
	{
		delete unit_x;
		unit_x = NULL;
	}
	if (unit_y)
	{
		delete unit_y;
		unit_y = NULL;
	}
	if (x)
	{
		unit_x = new char[strlen(x)+1];
		strcpy(unit_x, x);
	}
	if (y)
	{
		unit_y = new char[strlen(y)+1];
		strcpy(unit_y, y);
	}
}

void CGraph::set_maxcnt(int cnt)
{
	limit = cnt;
}

bool CGraph::on_expose_event(GdkEventExpose* event)
{
  // This is where we draw on the window
  Glib::RefPtr<Gdk::Window> window = get_window();
  if (window)
  {
    Cairo::RefPtr<Cairo::Context> cr = window->create_cairo_context();

    // clip to the area indicated by the expose event so that we only redraw
    // the portion of the window that needs to be redrawn
    cr->rectangle(event->area.x, event->area.y,
            event->area.width, event->area.height);
    cr->clip();

    // draw the graph
	draw_axes(cr);
	draw(cr, buffer, count, size);
  }

  return true;
}
